<?php

namespace Pepper\Money\Controller\Checkout;

use Magento\Framework\App\Action\Context;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Customer\Model\ResourceModel\Customer\CollectionFactory as CustomerFactory;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderFactory;
use Magento\Quote\Model\ResourceModel\Quote\CollectionFactory as QuoteFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductFactory;
use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Framework\Controller\Result\JsonFactory;

use Magento\Catalog\Model\ProductRepository;
use Magento\Checkout\Model\Cart;
use Magento\Checkout\Model\Session;
use Magento\Framework\Data\Form\FormKey;
use Magento\Framework\View\Result\PageFactory;
use Magento\Sales\Model\Order\Email\Sender\InvoiceSender;
use Magento\Sales\Model\Service\InvoiceService;
use Magento\Sales\Model\Order\Payment\Transaction;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Quote\Api\CartManagementInterface;

use Pepper\Money\Controller\PepperController;
use Pepper\Money\lib\Pepper\Client;

class Data extends \Magento\Framework\App\Action\Action
{
    protected $_jsonHelper;
    protected $_customerFactory;
    protected $_orderFactory;
    protected $_quoteFactory;
    protected $_jsonFactory;
    protected $_productFactory;
    protected $_imageHelper;
    protected $_PepperController;

    protected $currentOrderId;
    protected $currentCartId;
    protected $currentCustomerId;
    protected $allowedData;

    public function __construct(
        Context $context,
        Json $jsonHelper,
        CustomerFactory $customerFactory,
        OrderFactory $orderFactory,
        QuoteFactory $quoteFactory,
        JsonFactory $jsonFactory,
        ProductFactory $productFactory,
		ImageHelper $imageHelper,
		PepperController $PepperController
    )
    {
        $this->_jsonHelper = $jsonHelper;
        $this->_customerFactory = $customerFactory;
        $this->_orderFactory = $orderFactory;
        $this->_quoteFactory = $quoteFactory;
        $this->_jsonFactory = $jsonFactory;
        $this->_productFactory = $productFactory;
        $this->_imageHelper = $imageHelper;
    	$this->_PepperController = $PepperController;

        parent::__construct($context);
    }

    public function execute()
    {
		header('Content-type:application/json;charset=utf-8');
        $result = $this->_jsonFactory->create();
		$pepper_client = $this->_PepperController->getPepperClient();

        $request = (array)json_decode(file_get_contents('php://input'));
        if ($request == null) {
            http_response_code(400);
            $result->setData(['error' => 'Missing request parameters']);
            return $result;
        }

        $signature = $request['signature'];        
        $this->currentCartId = $request['cart_id'];
        $this->currentCustomerId = $request['customer_id'];
        $this->allowedData = $request['requested_data'];

        if ($this->currentCartId == null || $this->currentCustomerId == null) {
            http_response_code(400);
            $result->setData(['error' => 'Missing request parameters']);
            return $result;
        }

        if (!$pepper_client->checkMacroDataSignature($request, $signature)) {
            http_response_code(401);
            $result->setData(['error' => 'Wrong signature']);
            return $result;
        }
        
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $quote = $objectManager->create('\Magento\Quote\Model\Quote')->load($this->currentCartId);
        if (!$quote && !is_object($quote) || $quote->getId() === null) {
            http_response_code(404);
            $result->setData(['error' => sprintf('Unable to load cart by card id "%d".', $this->currentCartId)]);
            return $result;
        }
        
        $dataToTransfer = $this->getData();
        $result->setData($dataToTransfer);
        return $result;
    }

    private function getData()
    {
        $resultData = array();

        foreach ($this->allowedData as $data => $allowed)
        {
            if ($allowed)
            {
                switch ($data)
                {
                    case $data == 'client_data':
                        $resultData = array_merge($resultData, $this->getClientData());
                        break;
                    case $data == 'historical':
                        $resultData = array_merge($resultData, $this->getHistoricalData());
                        break;
                    case $data == 'lost_carts':
                        $resultData = array_merge($resultData, $this->getLostCartsData());
                        break;
                    case $data == 'images':
                        $resultData = array_merge($resultData, $this->getImagesData());
                        break;
                    default:
                        break;
                }
            }
        }

        return $resultData;
    }

    /**
     * @param $customerId
     * @return array
     * Auxiliary method to get all client data
     */
    private function getClientData()
    {
        $customerCollection = $this->getCustomerCollection()
            ->addFieldToFilter('entity_id', array('eq' => $this->currentCustomerId));
        $currentCustomer = $customerCollection->getFirstItem();

        $clientData = array(
            'entity_id' => intval($currentCustomer->getId()),
            'website_id' => intval($currentCustomer->getWebsiteId()),
            'email' => $currentCustomer->getEmail(),
            'group_id' => intval($currentCustomer->getGroupId()),
            'store_id' => intval($currentCustomer->getStoreId()),
            'created_at' => $currentCustomer->getCreatedAt(),
            'updated_at' => $currentCustomer->getUpdatedAt(),
            'is_active' => $currentCustomer->getIsActive(),
            'dob' => $currentCustomer->getDob(),
            'gender' => $currentCustomer->getGender(),
            'firstname' => $currentCustomer->getFirstname(),
            'lastname' => $currentCustomer->getLastname()
        );

        if ($clientData != null) {
            return array('client_data' => $clientData);
        } else {
            return array('client_data' => array());
        }
    }

    private function getHistoricalData()
    {
        $orderCollection = $this->getOrderCollection()
            ->addFieldToFilter('customer_id', array('eq' => $this->currentCustomerId));

        $historicalData = array();

        foreach ($orderCollection as $order) {
            $orderData = array(
                'entity_id' => intval($order->getId()),
                'increment_id' => intval($order->getIncrementId()),
                'state' => $order->getState(),
                'store_id' => intval($order->getStoreId()),
                'customer_id' => intval($order->getCustomerId()),
                'subtotal' => (float)round($order->getSubtotal(), 2),
                'grand_total' => (float)round($order->getGrandTotal(), 2),
                'discount_amount' => (float)round($order->getDisctountAmount(),2 ),
                'shipping_amount' => (float)round($order->getShippingAmount(), 2),
                'created_at' => $order->getCreatedAt(),
                'updated_at' => $order->getUpdatedAt(),
                'products' => array()
            );

            foreach ($order->getItems() as $item) {
                if ($item->getProductType() == 'configurable') continue;

                $itemData = array(
                    // 'item_id' => intval($item->getItemId()),
                    // 'order_id' => intval($item->getOrderId()),
                    'product_id' => intval($item->getProductId()),
                    'name' => $item->getName(),
                    'price' => (float)round($item->getPrice(), 2),
                    'quantity' => (float)round($item->getQtyOrdered(), 2)
                );

                array_push($orderData['products'], $itemData);
            }

            array_push($historicalData, $orderData);
        }

        if ($historicalData != null){
            return array('historical_data' => $historicalData);
        } else {
            return array('historical_data' => array());
        }
    }

    private function getLostCartsData()
    {
        $quoteCollection = $this->getQuoteCollection()
            ->addFieldToFilter('customer_id', array('eq' => $this->currentCustomerId))
            ->addFieldToFilter('entity_id', array('neq' => $this->currentCartId));

        $lostCartsData = array();

        foreach ($quoteCollection as $cart) {
            if ($cart->getId() == $this->currentCartId) {
                continue;
            }

            $cartData = array(
                'entity_id' => intval($cart->getId()),
                'store_id' => intval($cart->getStoreId()),
                'created_at' => $cart->getCreatedAt(),
                'updated_at' => $cart->getUpdatedAt(),
                'products' => array()
            );

            $cartItems = $cart->getAllItems();

            if (!$cartItems) continue;

            foreach ($cartItems as $item){
                if ($item->getProductType() == 'configurable') continue;

                $itemData = array(
                    'product_id' => intval($item['product_id']),
                    'price' => (float)round($item['price'], 2),
                    'quantity' => (float)round($item['qty'], 2)
                );

                array_push($cartData['products'], $itemData);
            }

            array_push($lostCartData, $cartData);
        }

        if ($lostCartsData != null) {
            return array('lost_carts' => $lostCartsData);
        } else {
            return array('lost_carts' => array());
        }
    }

    private function getImagesData() {
        $imagesArray = array();

        $orderItemsCollection = $this->getQuoteCollection()
            ->addFieldToFilter('entity_id', array('eq' => $this->currentCartId))
            ->getFirstItem()
            ->getAllItems();

        foreach ($orderItemsCollection as $item) {
            $product = $this->getProductCollection()
                ->addFieldToFilter('entity_id', array('eq' => $item->getProductId()));

            if ($product->getProductTypeIds()[0] == "configurable") continue;

            $image64 = base64_encode(file_get_contents($this->_imageHelper->init($product, 'product_base_image')->getUrl()));

            array_push($imagesArray, array(
                'id' => $item->getProductId(),
                'imageB64' => $image64
            ));
        }

        if ($imagesArray != '') {
            return array('images' => $imagesArray);
        } else {
            return array('images' => array());
        }
    }

    private function getCustomerCollection()
    {
        return $this->_customerFactory->create();
    }

    private function getOrderCollection()
    {
        return $this->_orderFactory->create();
    }

    private function getQuoteCollection()
    {
        return $this->_quoteFactory->create();
    }

    private function getProductCollection()
    {
        return $this->_productFactory->create();
    }
	
}
