<?php
/**
 * Pepper Module
 *
 * Copyright (c) 2020 Pepper
 *
 * @category  Payment
 * @author    Pepper
 * @copyright 2020, Pepper
 * @link      https://peppermoney.es/
 * @license   http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 * Description:
 *
 * Plugin de Prestashop para conectar con la pasarela de Pepper.
 *
 * --
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to soporte@peppermoney.es so we can send you a copy immediately.
 */

if (!defined('_PS_VERSION_')) {
    exit();
}

$autoloader_param = dirname(__FILE__) . '/lib/Pepper/Client.php';
try {
    require_once $autoloader_param;
} catch (\Exception $e) {
    throw new \Exception('Error en el plugin de Pepper al cargar la librería.');
}

class Pepper extends PaymentModule
{
    private $html = '';
    private $refund_status = 0;

    /**
     * Build module
     *
     * @see PaymentModule::__construct()
     */
    public function __construct()
    {
        $this->name = 'pepper';
        $this->tab = 'payments_gateways';
        $this->version = '0.4.0';
        $this->author = 'Pepper';
        $this->module_key = 'e79e02f3f8334d0caf773a808e2dc16e';
        $this->currencies = true;
        $this->currencies_mode = 'radio';
        $this->is_eu_compatible = 1;
        $this->controllers = array(
            'payment'
        );
        parent::__construct();
        $this->page = basename(__FILE__, '.php');
        $this->displayName = $this->l('Pepper');
        $this->description = $this->l('Plugin de Prestashop para conectar con la pasarela de Pepper.');
        $this->confirmUninstall = $this->l('¿Estás seguro que deseas eliminar tus detalles?');

        /* Add configuration warnings if needed */
        if (!Configuration::get('pepper_title')
            || !Configuration::get('pepper_api_url')
            || !Configuration::get('pepper_api_key')
            || !Configuration::get('pepper_public_key')
            || !Configuration::get('pepper_secret_key')
            || !Configuration::get('pepper_environment')
            || !Configuration::get('pepper_integration')
            || !Configuration::get('pepper_description')
            || !Configuration::get('pepper_payment_status')
            || !Configuration::get('pepper_logo_width')
            || !Configuration::get('pepper_enable_n')
            || !Configuration::get('pepper_title_n')
            || !Configuration::get('pepper_description_n')
            || !Configuration::get('pepper_calc_min_amount')
            || !Configuration::get('pepper_calc_max_amount')) {
            $this->warning = $this->l('Configuración del módulo incompleta.');
        }
    }

    /**
     * Install module
     *
     * @see PaymentModule::install()
     */
    public function install()
    {
        $hookHeader = 'header';
        $hookBackOfficeHeaderHeader = 'backOfficeHeader';
        $hookPaymentReturn = 'paymentReturn';
        /* If PS version is >= 8.0 */
        if (version_compare(_PS_VERSION_, '8.0', '>=')) {
            $hookHeader = 'displayHeader';
			$hookBackOfficeHeaderHeader = 'displayBackOfficeHeader';
			$hookPaymentReturn = 'displayPaymentReturn';
        }
        if (!parent::install()
            || !Configuration::updateValue('pepper_title', 'Paga tu compra a plazos con Pepper')
            || !Configuration::updateValue('pepper_api_url', 'https://api-ecommerce.peppermoney.es/v1/')
            || !Configuration::updateValue('pepper_api_key', '')
            || !Configuration::updateValue('pepper_public_key', '')
            || !Configuration::updateValue('pepper_secret_key', '')
            || !Configuration::updateValue('pepper_environment', 'PRO')
            || !Configuration::updateValue('pepper_integration', 'DEFAULT')
            || !Configuration::updateValue('pepper_description', '¿Y si lo pagas a plazos? Compra ahora y divide el coste en cuotas que se cargarán automáticamente cada mes en tu tarjeta. Sin sorpresas, papeleos ni complicaciones.')
            || !Configuration::updateValue('pepper_payment_status', _PS_OS_PAYMENT_)
            || !Configuration::updateValue('pepper_logo_width', '100')
            || !Configuration::updateValue('pepper_enable_n', 'NO')
            || !Configuration::updateValue('pepper_title_n', 'Divide tu compra en {{NumCuotas}} cuotas con Pepper')
            || !Configuration::updateValue('pepper_description_n', 'Divide el pago de tu compra en cuotas que se cargarán automáticamente cada mes en tu tarjeta. ¡Así de fácil!')
            || !Configuration::updateValue('pepper_calc_min_amount', -1.0)
            || !Configuration::updateValue('pepper_calc_max_amount', -1.0)
            || !$this->registerHook('payment')
            || !$this->registerHook($hookPaymentReturn)
            || !$this->registerHook('paymentOptions')
            || !$this->registerHook($hookHeader)
            || !$this->registerHook('displayProductPriceBlock')
            || !$this->registerHook($hookBackOfficeHeaderHeader)) {
            return false;
        }
        return true;
    }

    /**
     * Uninstall module
     *
     * @see PaymentModule::uninstall()
     */
    public function uninstall()
    {
        if (!Configuration::deleteByName('pepper_title')
            || !Configuration::deleteByName('pepper_api_url')
            || !Configuration::deleteByName('pepper_api_key')
            || !Configuration::deleteByName('pepper_public_key')
            || !Configuration::deleteByName('pepper_secret_key')
            || !Configuration::deleteByName('pepper_environment')
            || !Configuration::deleteByName('pepper_integration')
            || !Configuration::deleteByName('pepper_description')
            || !Configuration::deleteByName('pepper_payment_status')
            || !Configuration::deleteByName('pepper_logo_width')
            || !Configuration::deleteByName('pepper_enable_n')
            || !Configuration::deleteByName('pepper_title_n')
            || !Configuration::deleteByName('pepper_description_n')
            || !Configuration::deleteByName('pepper_calc_min_amount')
            || !Configuration::deleteByName('pepper_calc_max_amount')
            || !parent::uninstall()) {
            return false;
        }
        return true;
    }

    /**
     * Validate submited data
     */
    private function postValidation()
    {
        $this->_errors = array();
        if (Tools::getValue('submitUpdate')) {
            if (!Tools::getValue('title')) {
                $this->_errors[] = $this->l('Título es requerido.');
            }
            if (!Tools::getValue('api_url')) {
                $this->_errors[] = $this->l('URL de API es requerida.');
            }
            if (!Tools::getValue('api_key')) {
                $this->_errors[] = $this->l('API Key es requerido.');
            }
            if (!Tools::getValue('public_key')) {
                $this->_errors[] = $this->l('Clave pública es requerida.');
            }
            if (!Tools::getValue('secret_key')) {
                $this->_errors[] = $this->l('Clave secreta es requerida.');
            }
            if (!Tools::getValue('environment')) {
                $this->_errors[] = $this->l('Entorno es requerido.');
            }
            if (!Tools::getValue('integration')) {
                $this->_errors[] = $this->l('Forma de integración es requerido.');
            }
            if (!Tools::getValue('description')) {
                $this->_errors[] = $this->l('Descripción es requerida.');
            }
            if (!Tools::getValue('payment_status')) {
                $this->_errors[] = $this->l('El estado de pago es requerido.');
            }
            if (!Tools::getValue('logo_width')) {
                $this->_errors[] = $this->l('Ancho de logo es requerido.');
            }
            if (!Tools::getValue('enable_n')) {
                $this->_errors[] = $this->l('Habilitación o no de la opción Divide N es requerida.');
            }
            if (!Tools::getValue('title_n')) {
                $this->_errors[] = $this->l('Título de Divide N es requerido.');
            }
            if (!Tools::getValue('description_n')) {
                $this->_errors[] = $this->l('Descripción de Divide N es requerido.');
            }
        }
    }

    /**
     * Update submited configurations
     */
    public function getContent()
    {
        $this->html = '<h2>' . $this->displayName . '</h2>';
        if (Tools::isSubmit('submitUpdate')) {
            Configuration::updateValue('pepper_title', Tools::getValue('title'));
            Configuration::updateValue('pepper_api_url', Tools::getValue('api_url'));
            Configuration::updateValue('pepper_api_key', Tools::getValue('api_key'));
            Configuration::updateValue('pepper_public_key', Tools::getValue('public_key'));
            Configuration::updateValue('pepper_secret_key', Tools::getValue('secret_key'));
            Configuration::updateValue('pepper_environment', Tools::getValue('environment'));
            Configuration::updateValue('pepper_integration', Tools::getValue('integration'));
            Configuration::updateValue('pepper_description', Tools::getValue('description'));
            Configuration::updateValue('pepper_payment_status', Tools::getValue('payment_status') ? Tools::getValue('payment_status') : _PS_OS_PAYMENT_);
            Configuration::updateValue('pepper_logo_width', Tools::getValue('logo_width'));
            Configuration::updateValue('pepper_enable_n', Tools::getValue('enable_n'));
            Configuration::updateValue('pepper_title_n', Tools::getValue('title_n'));
            Configuration::updateValue('pepper_description_n', Tools::getValue('description_n'));
            Configuration::updateValue('pepper_calc_min_amount', (!Tools::getValue('calc_min_amount') ? -1.00 : (is_numeric(Tools::getValue('calc_min_amount')) ? floatval(Tools::getValue('calc_min_amount')) : '0' )));
            Configuration::updateValue('pepper_calc_max_amount', (!Tools::getValue('calc_max_amount') ? -1.00 : (is_numeric(Tools::getValue('calc_max_amount')) ? floatval(Tools::getValue('calc_max_amount')) : '0' )));
        }

        $this->postValidation();
        if (isset($this->_errors) && count($this->_errors)) {
            foreach ($this->_errors as $err) {
                $this->html .= $this->displayError($err);
            }
        } elseif (Tools::getValue('submitUpdate') && !count($this->_errors)) {
            $this->html .= $this->displayConfirmation($this->l('Configuración actualizada'));
        }

        return $this->html . $this->displayForm();
    }

    /**
     * Build and display admin form for configurations
     */
    private function displayForm()
    {
        $dfl = array(
            'action' => $_SERVER['REQUEST_URI'],
            'img_path' => $this->_path . 'views/img/icons/pepper.svg',
            'path' => $this->_path
        );

        $config = Configuration::getMultiple(array(
            'pepper_title',
            'pepper_api_url',
            'pepper_api_key',
            'pepper_public_key',
            'pepper_secret_key',
            'pepper_environment',
            'pepper_integration',
            'pepper_description',
            'pepper_logo_width',
            'pepper_enable_n',
            'pepper_title_n',
            'pepper_description_n',
            'pepper_calc_min_amount',
            'pepper_calc_max_amount'
        ));
		
		$pepper_payment_status = Configuration::get('pepper_payment_status') ? Configuration::get('pepper_payment_status') : _PS_OS_PAYMENT_;
		
		$orderStatuses = [];
        $prestashopOrderStatuses = OrderState::getOrderStates(Context::getContext()->language->id);

        foreach ($prestashopOrderStatuses as $prestashopOrderStatus) {
            $orderStatuses[] = [
                'value' => $prestashopOrderStatus['id_order_state'],
                'title' => $prestashopOrderStatus['name'],
            ];
        }

        $this->context->smarty->assign(array(
            'pepper' => array(
                'dfl' => $dfl,
                'config' => $config,
				'status' => $orderStatuses,
				'pepper_payment_status' => $pepper_payment_status
            )
        ));

        return $this->display(__FILE__, 'views/templates/admin/display_form.tpl');
    }

    /**
     * Build and display payment button
     *
     * @param unknown $params
     * @return boolean|\PrestaShop\PrestaShop\Core\Payment\PaymentOption[]
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->isPayment()) {
            return false;
        }

        $paymentLink = $this->context->link->getModuleLink($this->name, 'payment', array(
            'token' => Tools::getToken(false)
        ), true);
        $paymentLinkN = $this->context->link->getModuleLink($this->name, 'payment', array(
            'token' => Tools::getToken(false),
            'n' => 'SI'
        ), true);

        $this->context->smarty->assign('path', $this->_path);
        $this->context->smarty->assign('static_token', Tools::getToken(false));
        $this->context->smarty->assign('array_token', array('token' => Tools::getToken(false)));
        $this->context->smarty->assign('title', Configuration::get('pepper_title'));
        $this->context->smarty->assign('description', Configuration::get('pepper_description'));
        $this->context->smarty->assign('title_n', Configuration::get('pepper_title_n'));
        $this->context->smarty->assign('description_n', Configuration::get('pepper_description_n'));
        $this->context->smarty->assign('logo_width', Configuration::get('pepper_logo_width'));
        $this->context->smarty->assign('integration', Configuration::get('pepper_integration'));
        $this->context->smarty->assign('payment_status', Configuration::get('pepper_payment_status') ? Configuration::get('pepper_payment_status') : _PS_OS_PAYMENT_);
        $this->context->smarty->assign('calc_min_amount', Configuration::get('pepper_calc_min_amount'));
        $this->context->smarty->assign('calc_max_amount', Configuration::get('pepper_calc_max_amount'));
        $this->context->smarty->assign('payment_link', $paymentLink);
        $this->context->smarty->assign('payment_link_n', $paymentLinkN);

        $paymentOption = new \PrestaShop\PrestaShop\Core\Payment\PaymentOption();
        $paymentOption->setCallToActionText(Configuration::get('pepper_title'))
            ->setModuleName($this->name)
            ->setAdditionalInformation($this->context->smarty->fetch(
                'module:pepper/views/templates/hook/payment_options.tpl'
            ))
            ->setLogo($this->_path.'views/img/icons/pepper.svg');
        
        if (Configuration::get('pepper_integration') == 'DEFAULT') {
            $paymentOption->setAction($paymentLink);
        } else {
            $paymentOption->setAction('#peppermodal')
                ->setForm($this->context->smarty->fetch(
                    'module:pepper/views/templates/hook/payment_options_form.tpl'
                ));
        }

        $paymentOptions = array($paymentOption);

        if (Configuration::get('pepper_enable_n') == 'SI') {
            $paymentOptionDivideN = new \PrestaShop\PrestaShop\Core\Payment\PaymentOption();
            $paymentOptionDivideN->setCallToActionText(Configuration::get('pepper_title_n'))
                ->setModuleName($this->name)
                ->setAdditionalInformation($this->context->smarty->fetch(
                    'module:pepper/views/templates/hook/payment_options_n.tpl'
                ))
                ->setLogo($this->_path.'views/img/icons/pepper.svg');
            
            if (Configuration::get('pepper_integration') == 'DEFAULT') {
                $paymentOptionDivideN->setAction($paymentLinkN);
            } else {
                $paymentOptionDivideN->setAction('#peppermodaln')
                    ->setForm($this->context->smarty->fetch(
                        'module:pepper/views/templates/hook/payment_options_form_n.tpl'
                    ));
            }
            array_push($paymentOptions, $paymentOptionDivideN);
        }

        return $paymentOptions;
    }

    /**
     * Build and display payment button
     *
     * @param array $params
     * @return string Templatepart
     */
    public function hookPayment($params)
    {
        if (!$this->isPayment()) {
            return false;
        }

        $this->context->smarty->assign('path', $this->_path);
        $this->context->smarty->assign('static_token', Tools::getToken(false));
        $this->context->smarty->assign('array_token', array('token' => Tools::getToken(false)));
        $this->context->smarty->assign('title', Configuration::get('pepper_title'));
        $this->context->smarty->assign('description', Configuration::get('pepper_description'));
        $this->context->smarty->assign('enable_n', Configuration::get('pepper_enable_n'));
        $this->context->smarty->assign('title_n', Configuration::get('pepper_title_n'));
        $this->context->smarty->assign('description_n', Configuration::get('pepper_description_n'));
        $this->context->smarty->assign('logo_width', Configuration::get('pepper_logo_width'));
        $this->context->smarty->assign('integration', Configuration::get('pepper_integration'));
        $this->context->smarty->assign('payment_status', Configuration::get('pepper_payment_status') ? Configuration::get('pepper_payment_status') : _PS_OS_PAYMENT_);
        $this->context->smarty->assign('payment_link', $this->context->link->getModuleLink($this->name, 'payment', array(
            'token' => Tools::getToken(false)
        ), true));
        $this->context->smarty->assign('payment_link_n', $this->context->link->getModuleLink($this->name, 'payment', array(
            'token' => Tools::getToken(false),
            'n' => 'SI'
        ), true));

        return $this->display(__FILE__, 'views/templates/hook/payment.tpl');
    }

    /**
     * Build and display confirmation
     *
     * @param array $params
     * @return string Templatepart
     */
    public function hookPaymentReturn($params)
    {
        if (!$this->isPayment()) {
            return false;
        }

        $this->context->smarty->assign('path', $this->_path);

        /* If PS version is >= 1.7 */
        if (version_compare(_PS_VERSION_, '8.0', '>=')) {
			$curr = new Currency($params['order']->id_currency);
            $this->context->smarty->assign(array(
                'amount' => Context::getContext()->currentLocale->formatPrice($params['order']->getOrdersTotalPaid(), $curr->iso_code)
            ));
        } elseif (version_compare(_PS_VERSION_, '1.7', '>=')) {
            $this->context->smarty->assign(array(
                'amount' => Tools::displayPrice(
                    $params['order']->getOrdersTotalPaid(),
                    new Currency($params['order']->id_currency),
                    false
                )
            ));
        } else {
            $this->context->smarty->assign(array(
                'amount' => Tools::displayPrice(
                    $params['total_to_pay'],
                    $params['currencyObj'],
                    false
                )
            ));
        }

        $this->context->smarty->assign('shop_name', $this->context->shop->name);

        return $this->display(__FILE__, 'views/templates/hook/payment_return.tpl');
    }
    public function hookDisplayPaymentReturn($params)
    {
        if (!$this->isPayment()) {
            return false;
        }

        $this->context->smarty->assign('path', $this->_path);

        /* If PS version is >= 1.7 */
        if (version_compare(_PS_VERSION_, '8.0', '>=')) {
			$curr = new Currency($params['order']->id_currency);
            $this->context->smarty->assign(array(
                'amount' => Context::getContext()->currentLocale->formatPrice($params['order']->getOrdersTotalPaid(), $curr->iso_code)
            ));
        } elseif (version_compare(_PS_VERSION_, '1.7', '>=')) {
            $this->context->smarty->assign(array(
                'amount' => Tools::displayPrice(
                    $params['order']->getOrdersTotalPaid(),
                    new Currency($params['order']->id_currency),
                    false
                )
            ));
        } else {
            $this->context->smarty->assign(array(
                'amount' => Tools::displayPrice(
                    $params['total_to_pay'],
                    $params['currencyObj'],
                    false
                )
            ));
        }

        $this->context->smarty->assign('shop_name', $this->context->shop->name);

        return $this->display(__FILE__, 'views/templates/hook/payment_return.tpl');
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function addHeaderJS()
    {
        Media::addJsDef(
            array(
                'pepper_api_url' => Configuration::get('pepper_api_url'),
                'pepper_public_key' => Configuration::get('pepper_public_key'),
                'pepper_api_key' => Configuration::get('pepper_api_key'),
                'pepper_environment' => Configuration::get('pepper_environment'),
                'pepper_calc_min_amount' => floatval(Configuration::get('pepper_calc_min_amount')),
                'pepper_calc_max_amount' => floatval(Configuration::get('pepper_calc_max_amount'))
            )
        );
        $this->context->controller->addJS($this->_path . '/views/js/front.js');
    }

    public function hookHeader()
    {
        $this->addHeaderJS();
    }

    public function hookDisplayHeader()
    {
        $this->addHeaderJS();
    }

    /** */
    public function hookDisplayProductPriceBlock($params)
    {
        //only hook beneath price block on product page
        if (isset($params['hook_origin']) && $params['hook_origin'] === 'product_sheet' && $params['type'] === 'after_price') {

            $lang_id  = $this->context->language->id; // get instance language id
            // $currency = $this->context->currency->sign; // get instance currency sign

            $id_product = (int) Tools::getValue('id_product'); // get current product page product id
            $product    = new Product($id_product, false, $lang_id);

            $price = $product->getPrice(true);
            $price = number_format($price, 2); // format new price to 2 decimal places

            $this->context->smarty->assign('price', $price);

            return $this->display(__FILE__, 'views/templates/hook/product.tpl');
        }
    }
    public function hookBackOfficeHeader()
    {
		if (Tools::getValue('controller') == "AdminOrders") {
			$this->context->controller->addJS($this->_path . '/views/js/back.js');
        }
    }
    public function hookDisplayBackOfficeHeader()
    {
		if (Tools::getValue('controller') == "AdminOrders") {
			$this->context->controller->addJS($this->_path . '/views/js/back.js');
        }
    }
    /**
     * Check if payment is active
     *
     * @return boolean
     */
    public function isPayment()
    {
        if (!$this->active) {
            return false;
        }

        if (!Configuration::get('pepper_title')
            || !Configuration::get('pepper_api_url')
            || !Configuration::get('pepper_public_key')
            || !Configuration::get('pepper_secret_key')
            || !Configuration::get('pepper_description')
            || !Configuration::get('pepper_logo_width')
            || !Configuration::get('pepper_enable_n')
            || !Configuration::get('pepper_title_n')
            || !Configuration::get('pepper_description_n')) {
            return false;
        }

        return true;
    }

    protected function get_client_config() {
        return array(
            'public_key' => Configuration::get('pepper_public_key'),
            'secret_key' => Configuration::get('pepper_secret_key'),
            'api_url' => Configuration::get('pepper_api_url')
        );
    }
}
