<?php
/**
 * Pepper Module
 *
 * Copyright (c) 2020 Pepper
 *
 * @category  Payment
 * @author    Pepper
 * @copyright 2020, Pepper
 * @link      https://peppermoney.es/
 * @license   http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 * Description:
 *
 * Plugin de Prestashop para conectar con la pasarela de Pepper.
 *
 * --
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to soporte@peppermoney.es so we can send you a copy immediately.
 */

require_once dirname(__FILE__) . '/../../config/config.inc.php';
require_once dirname(__FILE__) . '/../../init.php';
require_once dirname(__FILE__) . '/pepper.php';
require_once dirname(__FILE__) . '/lib/Pepper/Client.php';

function get_client_config() {
    return array(
        'public_key' => Configuration::get('pepper_public_key'),
        'secret_key' => Configuration::get('pepper_secret_key'),
        'api_url' => Configuration::get('pepper_api_url')
    );
}

function changeOrderStatus() {
    $config = get_client_config();
    $pepper_client = new Pepper\Client($config['public_key'], $config['secret_key'], $config['api_url']);

    $data = (array)json_decode(file_get_contents('php://input'));
    $status = $data['result'];
    $cart_id = $data['cart_id'];
    $order = $data['order'];
    $pepper_order_id = $order->order_id;
    $total_price = $order->total_price;
    $signature = $data['signature'];

    $context = Context::getContext();
    $pepper = new Pepper();
    $url = array(
        'cancel' => $context->shop->getBaseURL(true) . 'index.php?controller=order&step=3'
    );

    if ($status && $cart_id && $pepper_order_id) {
    } else {
        http_response_code(400);
        $result = array();
        $result['result'] = 'failure';
        $result['message'] = 'Params result, cart_id, order are required';
        $result['redirect_url'] = $url['cancel'];
        return $result;
    }
    if (!$pepper_client->checkSignature($data, $signature)) {
        http_response_code(401);
        $result = array();
        $result['result'] = 'failure';
        $result['message'] = 'Wrong signature';
        $result['redirect_url'] = $url['cancel'];
        return $result;
    }

    $redirect_url = $url['cancel'];
    $message = '';
    $result = 'failure';

    try {
        if ((!$cart = new Cart((int) $cart_id)) || !is_object($cart) || $cart->id === null) {
            http_response_code(404);
            $result = array();
            $result['result'] = 'failure';
            $result['message'] = sprintf('Unable to load cart by card id "%d".', $cart_id);
            $result['redirect_url'] = $url['cancel'];
            return $result;
        }
        if ((!$customer = new Customer($cart->id_customer))) {
            http_response_code(404);
            $result = array();
            $result['result'] = 'failure';
            $result['message'] = 'Invalid or missing customer secure key for this transaction.';
            $result['redirect_url'] = $url['cancel'];
            return $result;
        }

        if ($total_price !== null && $total_price != $cart->getOrderTotal()) {
            $result = array();
            $result['result'] = 'FRAUD';
            $result['message'] = 'Price not equal, current price = ' . $cart->getOrderTotal() . ' and price from cart = ' . $total_price;
            $result['redirect_url'] = $url['cancel'];
            return $result;
        }
		$payment_status = _PS_OS_PAYMENT_;
		if (Configuration::get('pepper_payment_status') && !empty(Configuration::get('pepper_payment_status')))
		{
			$payment_status = Configuration::get('pepper_payment_status');
		}

        switch ($status) {
            case $status == 'OPE':
                $pepper->validateOrder(
                    $cart->id,
					$payment_status,
                    $cart->getOrderTotal(),
                    $pepper->displayName,
                    $pepper->l(sprintf('Pepper transaction ID: %s.', $pepper_order_id)),
                    array('transaction_id' => $pepper_order_id),
                    null,
                    false,
                    $customer->secure_key,
                    null
                );

                if (_PS_VERSION_ <= '1.5') {
                    $redirect_url = $context->shop->getBaseURL(true) . 'index.php?controller=order-confirmation&id_cart='.
                        $cart->id.'&id_module='.$pepper->id.'&id_order='.$pepper->currentOrder.'&key='.
                        $customer->secure_key;
                } else {
                    $redirect_url = $context->link->getPageLink('order-confirmation', null, null, array(
                        'id_cart' => $cart->id,
                        'id_module' => $pepper->id,
                        'key' => $customer->secure_key,
                        'id_order' => $pepper->currentOrder
                    ));
                }
                $result = 'success';
                $message = 'Successfully updated status from order with order id = ' . $pepper_order_id . ' and cart id = ' . $cart_id . ' to ' . $status . 'with payment_method ' . $payment_method;
                break;
            case $status == 'CAN':
                $result = 'success';
                $message = 'Status ignored';
                break;
            case $status == 'EXP':
                $result = 'success';
                $message = 'Status ignored';
                break;
            case $status == 'REJ':
                $result = 'success';
                $message = 'Status ignored';
                break;
            case $status == 'PEN':
                $result = 'success';
                $message = 'Your order will be processed when the payment is finished';
                break;
        }    
    } catch (Exception $e) {
        $message = 'Prestashop error: ' . $e->getMessage();
    }

    return [
        'result' => $result,
        'message' => $message,
        'redirect_url' => $redirect_url
    ];
}

$data = changeOrderStatus();

header('Content-type:application/json;charset=utf-8');
die(json_encode($data));
