<?php
/**
 * Plugin Name: Pepper WooCommerce Plugin
 * Plugin URI: https://developer.pepperfinance.es/plugins-woocommerce.html
 * Description: Plugin de WooCommerce para conectar con el API de Pepper.
 * Author: Pepper
 * Author URI: https://peppermoney.es/
 * Version: 0.2.5
 * Text Domain: wc_pepper
 * Domain Path: /i18n/languages/
 *
 * Copyright: (c) 2020 Pepper y WooCommerce
 *
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 *
 * @package   WC-Gateway-Pepper
 * @author    Pepper
 * @category  Admin
 * @copyright Copyright (c) 2020 Pepper y WooCommerce
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

defined( 'ABSPATH' ) or exit;
// WooCommerce is active
if ( ! in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    return;
}

/**
 * Add the gateway to WC Available Gateways
 *
 * @since 1.0.0
 * @param array $gateways all available WC gateways
 * @return array $gateways all WC gateways + pepper gateway
 */
function wc_pepper_add_to_gateways( $gateways ) {
    $gateways[] = 'WC_Gateway_Pepper';
    $gateways[] = 'WC_Gateway_Pepper_N';
    return $gateways;
}
add_filter( 'woocommerce_payment_gateways', 'wc_pepper_add_to_gateways' );

/**
 * Adds plugin page links
 *
 * @since 1.0.0
 * @param array $links all plugin links
 * @return array $links all plugin links + our custom links (i.e., "Settings")
 */
function wc_pepper_gateway_plugin_links( $links ) {
    $plugin_links = array(
        '<a href="' . admin_url( 'admin.php?page=wc-settings&tab=checkout&section=pepper_gateway' ) . '">' . __( 'Configurar', 'wc-gateway-pepper' ) . '</a>'
    );
    return array_merge( $plugin_links, $links );
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'wc_pepper_gateway_plugin_links' );

$autoloader_param = __DIR__ . '/lib/Pepper/Client.php';
try {
    require_once $autoloader_param;
} catch (\Exception $e) {
    throw new \Exception('Error en el plugin de Pepper al cargar la librería.');
}

/**
 * Pepper Payment Gateway
 *
 * Provides an Pepper Payment Gateway.
 * We load it later to ensure WC is loaded first since we're extending it.
 *
 * @class 		WC_Gateway_Pepper
 * @extends		WC_Payment_Gateway
 * @version		0.0.1
 * @package		WooCommerce/Classes/Payment
 * @author 		Pepper
 */

class WC_Pepper {

    /**
     * Constructor for the gateway.
     */
    public function __construct() {
        $this->enabled = true;
        add_action( 'wp_head', array( $this, 'pepper_js' ), 999999 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'product_widget' ), 100 );

        $this->api_url = NULL;
        $this->api_key = NULL;
        $this->public_key = NULL;
        $this->environment = NULL;

    }

    function get_pepper_template($template_name) {
        $template_path = WC()->template_path() . '/pepper-woocomerce/';
        $default_path  = plugin_dir_path( __FILE__ ) . 'templates/';

        if ( !$this->public_key ) {
            $settings = get_option('woocommerce_pepper_gateway_settings', null);
            if ( $settings && isset( $settings[ 'public_key' ] ) ) {
                $this->public_key = $settings[ 'public_key' ];
            }
        }

        if ( !$this->api_url ) {
            $settings = get_option('woocommerce_pepper_gateway_settings', null);
            if ( $settings && isset( $settings[ 'api_url' ] ) ) {
                $this->api_url = $settings[ 'api_url' ];
            }
        }

        if ( !$this->api_key ) {
            $settings = get_option('woocommerce_pepper_gateway_settings', null);
            if ( $settings && isset( $settings[ 'api_key' ] ) ) {
                $this->api_key = $settings[ 'api_key' ];
            }
        }

        if ( !$this->environment ) {
            $settings = get_option('woocommerce_pepper_gateway_settings', null);
            if ( $settings && isset( $settings[ 'environment' ] ) ) {
                $this->environment = $settings[ 'environment' ];
            }
        }

        wc_get_template( $template_name, array(
            'public_key' => $this->public_key,
            'api_url' => $this->api_url,
            'api_key' => $this->api_key,
            'environment' => $this->environment
        ), $template_path, $default_path );
    }

    function pepper_js() {
        $this->get_pepper_template( 'pepper-header.php' );
    }
        
    /*
     * Show widget in product page
    */
    public function product_widget() {
        $this->get_pepper_template( 'pepper-product.php' );
    }
}

add_action( 'plugins_loaded', 'wc_pepper_gateway_init', 11 );
function wc_pepper_gateway_init() {
    class WC_Gateway_Pepper extends WC_Payment_Gateway {
        /**
         * Constructor for the gateway.
         */
        public function __construct($is_base=TRUE) {

            $this->id                 = 'pepper_gateway';
            $this->icon               = apply_filters( 'woocommerce_pepper_icon', plugins_url( 'assets/images/icons/pepper.svg' , __FILE__ ) );
            $this->has_fields         = false;
            $this->method_title       = __( 'Pepper', 'wc-gateway-pepper' );
            $this->method_description = __( 'Permite utilizar el método de pago Pepper en tu sitio web.', 'wc-gateway-pepper' );

            // Load the settings.
            $this->init_form_fields();
            $this->init_settings();

            // Define user set variables
            $this->title = $this->get_option( 'title' );
            $this->api_url = $this->get_option( 'api_url' );
            $this->api_key = $this->get_option( 'api_key' );
            $this->public_key = $this->get_option( 'public_key' );
            $this->secret_key = $this->get_option( 'secret_key' );
            $this->environment = $this->get_option( 'environment', 'TST' );
            $this->integration = $this->get_option( 'integration', 'DEFAULT' );
            $this->description  = $this->get_option( 'description' );
            $this->thank_you_text = $this->get_option( 'thank_you_text', '' );
            $this->set_completed = $this->get_option( 'set_completed', 'N' );
            $this->icon = $this->get_option( 'icon', $this->icon );
            $this->logo_width  = $this->get_option( 'logo_width' );
            $this->enabled_n = $this->get_option('enabled_n');
            $this->title_n = $this->get_option('title_n');
            $this->description_n = $this->get_option('description_n');
            $this->is_standard = true;

            // Actions
            if ( $is_base ) {
                add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
                add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );
            }

            $this->notify_url = add_query_arg( 'wc-api', get_class( $this ), home_url( '/' ) );

            // Actions
            if ( $is_base ) {

                if ( version_compare( WOOCOMMERCE_VERSION, '2.0', '<' ) ) {
                    // Check for gateway messages using WC 1.X format
                    add_action( 'init', array( $this, 'check_notification' ) );
                    add_action( 'woocommerce_update_options_payment_gateways', array( &$this, 'process_admin_options' ) );
                } else {
                    // Payment listener/API hook (WC 2.X)
                    add_action( 'woocommerce_api_' . strtolower( get_class( $this ) ), array( $this, 'check_notification' ) );
                    add_action('woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
                }

                add_action( 'woocommerce_receipt_'.$this->id, array( $this, 'receipt_page' ) );
            }
            // Pepper icon with configuration from admin panel.
            add_filter( 'woocommerce_gateway_icon', array( $this, 'pepper_icon' ), 10, 2 );
        }

        function pepper_icon($icon, $id) {
            if ( $id === $this->id ) {
                return '<img src="' . plugins_url( 'assets/images/icons/pepper.svg' , __FILE__ ) . '" width="' . $this->logo_width .'" > ';
            } else {
                return $icon;
            }
        }

        function get_pepper_client() {
            return new Pepper\Client($this->public_key, $this->secret_key, $this->api_url);
        }

        /**
         * Initialize Gateway Settings Form Fields
         */
        public function init_form_fields() {

            $this->form_fields = apply_filters( 'wc_pepper_form_fields', array(

                'enabled' => array(
                    'title'   => __( 'Habilitar', 'wc-gateway-pepper' ),
                    'type'    => 'checkbox',
                    'label'   => __( 'Habilitar método de pago Pepper', 'wc-gateway-pepper' ),
                    'default' => 'yes'
                ),

                'api_url' => array(
                    'title'       => __( 'URL de API', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'URL de API dada por Pepper.', 'wc-gateway-pepper' ),
                    'default'     => 'https://api-ecommerce.peppermoney.es/v1/',
                    'desc_tip'    => true,
                ),

                'api_key' => array(
                    'title'       => __( 'API Key', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'API Key dada por Pepper.', 'wc-gateway-pepper' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),

                'public_key' => array(
                    'title'       => __( 'Clave Pública', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'Clave Pública dada por Pepper.', 'wc-gateway-pepper' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),

                'secret_key' => array(
                    'title'       => __( 'Clave Secreta', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'Clave secreta dada por Pepper.', 'wc-gateway-pepper' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),

                'environment' => array(
                    'title'       => __( 'Entorno', 'wc-gateway-pepper' ),
                    'type'        => 'select',
                    'description' => __( 'Entorno usado para mostrar los widgets.', 'wc-gateway-pepper' ),
                    'desc_tip'    => false,
                    'options'     => array(
                        'PRO' => __( 'Producción', 'wc-gateway-pepper' ),
                        'TST' => __( 'Test', 'wc-gateway-pepper' ),
                    ),
                    'default'     => 'TST'
                ),

                'integration' => array(
                    'title'       => __( 'Forma de integración', 'wc-gateway-pepper' ),
                    'type'        => 'select',
                    'description' => __( 'Define si se usará una redirección o iframe.', 'wc-gateway-pepper' ),
                    'desc_tip'    => false,
                    'options'     => array(
                        'DEFAULT' => __( 'Redirección', 'wc-gateway-pepper' ),
                        'IFRAME' => __( 'iFrame', 'wc-gateway-pepper' ),
                    ),
                    'default'     => 'DEFAULT'
                ),

                'title' => array(
                    'title'       => __( 'Título', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'Título mostrado al cliente durante el proceso de compra con este método de pago.', 'wc-gateway-pepper' ),
                    'default'     => __( 'Paga tu compra a plazos con Pepper', 'wc-gateway-pepper' ),
                    'desc_tip'    => true,
                ),

                'description' => array(
                    'title'       => __( 'Descripción', 'wc-gateway-pepper' ),
                    'type'        => 'textarea',
                    'description' => __( 'Descripción mostrada al cliente durante el proceso de compra con este método de pago.', 'wc-gateway-pepper' ),
                    'default'     => __( '¿Y si lo pagas a plazos? Compra ahora y divide el coste en cuotas que se cargarán automáticamente cada mes en tu tarjeta. Sin sorpresas, papeleos ni complicaciones.', 'wc-gateway-pepper' ),
                    'desc_tip'    => true,
                ),

                'thank_you_text' => array(
                    'title'       => __( 'Texto de la página de gracias', 'wc-gateway-pepper' ),
                    'type'        => 'textarea',
                    'description' => __( 'Texto que se agregará a la página de gracias.', 'wc-gateway-pepper' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'set_completed' => array(
                    'title'       => __( '¿Marcar el pedido como completado después del pago?', 'wc-gateway-pepper' ),
                    'type'        => 'select',
                    'description' => __( 'Después del pago, ¿debe mostrarse el pedido como completado? Por defecto es "processing".', 'wc-gateway-pepper' ),
                    'desc_tip'    => false,
                    'options'     => array(
                        'N' => __( 'No', 'wc-gateway-pepper' ),
                        'Y' => __( 'Si', 'wc-gateway-pepper' ),
                    ),
                    'default'     => 'N'
                ),

                'logo_width' => array(
                    'title'       => __( 'Ancho de Logo en Checkout', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'Debe incluir al final px o % o cualquier unidad de medida.', 'wc-gateway-pepper' ),
                    'default'     => '69px',
                    'desc_tip'    => true,
                ),
                // 'icon' => array(
                //     'title'   => __( 'Icon', 'wc-gateway-pepper' ),
                //     'type'    => 'text',
                //     'label'   => __( 'Url de la imagen a mostrar en la página de pago', 'wc-gateway-pepper' ),
                //     'default' => apply_filters( 'woocommerce_pepper_icon', plugins_url( 'assets/images/icons/pepperpayment.png' , __FILE__ ) )
                // ),
                
                'enabled_n' => array(
                    'title'   => __( 'Opción Divide N', 'wc-gateway-pepper' ),
                    'type'    => 'checkbox',
                    'label'   => __( 'Habilitar método de pago Pepper en N', 'wc-gateway-pepper' ),
                    'default' => 'no'
                ),

                'title_n' => array(
                    'title'       => __( 'Título Opción Divide N', 'wc-gateway-pepper' ),
                    'type'        => 'text',
                    'description' => __( 'Título mostrado al cliente durante el proceso de compra en la opción Divide N.', 'wc-gateway-pepper' ),
                    'default'     => __( 'Divide tu compra en {{NumCuotas}} cuotas con Pepper', 'wc-gateway-pepper' ),
                    'desc_tip'    => true,
                ),

                'description_n' => array(
                    'title'       => __( 'Descripción Opción Divide N', 'wc-gateway-pepper' ),
                    'type'        => 'textarea',
                    'description' => __( 'Descripción mostrada al cliente durante el proceso de compra en la opción Divide N.', 'wc-gateway-pepper' ),
                    'default'     => __( 'Divide el pago de tu compra en cuotas que se cargarán automáticamente cada mes en tu tarjeta. ¡Así de fácil!', 'wc-gateway-pepper' ),
                    'desc_tip'    => true,
                ),

            ) );
        }


        /**
         * Output for the order received page.
         */
        public function thankyou_page() {
            if ( $this->thank_you_text ) {
                echo wpautop( wptexturize( $this->thank_you_text ) );
            }
        }


        /**
         * Process the payment and return the result
         *
         * @param int $order_id
         * @return array
         */
        public function process_payment( $order_id ) {
            $order = wc_get_order( $order_id );

            $redirect_url = $order->get_checkout_payment_url(true);

            return array(
                    'result'        => 'success',
                    'redirect'      => $redirect_url
            );
        }

        function receipt_page( $order_id, $force = false ) {
            if ($force || $this->integration == 'DEFAULT') {
                echo '<p>'.__( 'Redirigiendo a Pepper.', 'wc-gateway-pepper' ).'</p>';
                $order = wc_get_order( $order_id );    
                $pepper_client = $this->get_pepper_client();    
                $result = $this->process_regular_payment( $pepper_client, $order, $order_id );
            } else {
                $template_path = WC()->template_path() . '/pepper-woocomerce/';
                $default_path  = plugin_dir_path( __FILE__ ) . 'templates/';
                $current_iframe = $this->is_standard ? 'pepper-iframe.php' : 'pepper-iframe_n.php';
                wc_get_template( $current_iframe, array(
                    'src' => $this->notify_url.'&iframe='.$order_id,
                    'checkout_url' => wc_get_checkout_url()
                ), $template_path, $default_path );
            }

            // Mark as on-hold (we're awaiting the payment)
            // $order->update_status( 'on-hold', __( 'Esperando la confirmación del pago por Pepper', 'wc-gateway-pepper' ) );

            if ( version_compare( WOOCOMMERCE_VERSION, '2.0', '<' ) ) {
                // $woocommerce->cart->empty_cart();
            } else {
                // WC()->cart->empty_cart();
            }
        }


        /**
         * Process regular payment and return the result
         *
         * @param object $pepper_client
         * @param object $order
         * @param int $order_id
         * @return array
         */
        public function process_regular_payment( $pepper_client, $order, $order_id ) {
            $json = $pepper_client->getCartData(
                $order, 
                $order_id, 
                $this->notify_url, 
                true, 
                $this->integration,
                $this->is_standard ? 'STD' : 'DVN'
            );
            echo $pepper_client->getForm($json);
        }

        /**
         * Check for Pepper notification
         *
         * @access public
         * @return void
         */
        function check_notification() {
            global $woocommerce;

            if(isset($_GET['iframe'])) {
                die($this->receipt_page($_GET['iframe'], true));
            }else if(isset($_GET['get-cart'])) {
                $data = $this->getCartData();
            } else if(isset($_GET['get-data'])) {
                $data = $this->gatherAllData();
            } else {
                $data = $this->changeOrderStatus();
            }

            header('Content-type:application/json;charset=utf-8');
            die(json_encode($data));
        }

        function getCartData() {
            $pepper_client = $this->get_pepper_client();

            $request = (array)json_decode(file_get_contents('php://input'));

            if ($request == null) {
                http_response_code(400);
                return ['error' => 'Missing request parameters'];
            }

            $signature = $request['signature'];       
            $id_cart = $request['cart_id'];

            if ($id_cart == null) {
                http_response_code(400);
                return ['error' => 'Missing request parameters'];
            }

            if (!$pepper_client->checkCartDataSignature($request, $signature)) {
                http_response_code(401);
                return ['error' => 'Wrong signature'];
            }

            $order = wc_get_order( $id_cart );
            if ( !$order ) {
                http_response_code(404);
                return ['error' => 'Cart ID does not match'];
            }

            if ( $order->has_status( 'completed' ) ) {
                http_response_code(400);
                return ['error' => 'Cart paid'];
            }
            
            $json = $pepper_client->getCartData(
                $order, 
                $id_cart, 
                $this->notify_url,
                false, 
                null,
                array_key_exists('n', $_GET) ? ($_GET['n'] == 'SI' ? 'DVN' : 'STD') : 'STD'
            );

            return $json;
        }

        function changeOrderStatus() {
            $pepper_client = $this->get_pepper_client();

            $data = (array)json_decode(file_get_contents('php://input'));
            $status = $data['result'];
            $cart_id = $data['cart_id'];
            $order = $data['order'];
            $pepper_order_id = $order->order_id;
            $total_price = $order->total_price;
            $signature = $data['signature'];

            if ($status && $cart_id && $pepper_order_id) {
            } else {
                http_response_code(400);
                $result = array();
                $result['result'] = 'failure';
                $result['message'] = __('Params result, cart_id, order are required', 'wc-gateway-pepper');
                $result['redirect_url'] = wc_get_cart_url();
                return $result;
            }

            if (!$pepper_client->checkSignature($data, $signature)) {
                http_response_code(401);
                $result = array();
                $result['result'] = 'failure';
                $result['message'] = __('Wrong signature', 'wc-gateway-pepper');
                $result['redirect_url'] = wc_get_cart_url();
                return $result;
            }

            $redirect_url = wc_get_cart_url();
            $message = '';
            $result = 'failure';

            try {
                $order = wc_get_order( $cart_id );

                if ( !$order ) {
                    http_response_code(404);
                    $result = array();
                    $result['result'] = 'failure';
                    $result['message'] = __('Cart ID does not match', 'wc-gateway-pepper');
                    $result['redirect_url'] = wc_get_cart_url();
                    return $result;
                }

                if ($total_price !== null && $total_price != (float)round($order->total, 2)) {
                    $result = array();
                    $result['result'] = 'FRAUD';
                    $result['message'] = __('Price not equal, current price = ' . $order->total . ' and price from cart = ' . $total_price, 'wc-gateway-pepper');
                    $result['redirect_url'] = wc_get_cart_url();
                    return $result;
                }

                if ( $order->has_status( 'completed' ) ) {
                    $message = __('Order status already completed', 'wc-gateway-pepper');
                } else {
                    switch ($status) {
                        case $status == 'OPE':
                            // Payment completed
                            $order->add_order_note( __('Pago completado por Pepper con referencia: '.$pepper_order_id, 'wc-gateway-pepper') );
                            $order->payment_complete( $pepper_order_id );

                            // Set order as completed if user did set up it
                            if ( 'Y' == $this->set_completed ) {
                                $order->update_status( 'completed' );
                            }

                            $result = 'success';
                            $message = __('Successfully updated status from order with order id = ' . $pepper_order_id . ' and cart id = ' . $cart_id . ' to ' . $status, 'wc-gateway-pepper');
                            $redirect_url = wc_get_endpoint_url( 'order-received', $order->get_id(), wc_get_page_permalink( 'checkout' ) );
			                $redirect_url = add_query_arg( 'key', $order->get_order_key(), $redirect_url );
                            break;
                        case $status == 'CAN':
                            $result = 'success';
                            $message = 'Status ignored';
                            break;
                        case $status == 'EXP':
                            $result = 'success';
                            $message = 'Status ignored';
                            break;
                        case $status == 'REJ':
                            $result = 'success';
                            $message = 'Status ignored';
                            break;
                        case $status == 'PEN':
                            $result = 'success';
                            $message = 'Your order will be processed when the payment is finished';
                            break;
                    }
                }    
            } catch (Exception $e) {
                $message = __('WooCommerce error: ' . $e->getMessage(), 'wc-gateway-pepper');
            }
    
            return [
                'result' => $result,
                'message' => $message,
                'redirect_url' => $redirect_url
            ];
        }

        function gatherAllData() {
            $pepper_client = $this->get_pepper_client();

            $checkOutInformation = (array)json_decode(file_get_contents('php://input'));

            if ($checkOutInformation == null) {
                http_response_code(400);
                return ['error' => 'Missing request parameters'];
            }

            $signature = $checkOutInformation['signature'];        
            $order_id = $checkOutInformation['cart_id'];
            $user_id = $checkOutInformation['customer_id'];

            if ($order_id == null) {
                http_response_code(400);
                return ['error' => 'Missing request parameters'];
            }

            if (!$pepper_client->checkMacroDataSignature($checkOutInformation, $signature)) {
                http_response_code(401);
                return ['error' => 'Wrong signature'];
            }
            
            $order = wc_get_order( $order_id );
            if ( !$order ) {
                http_response_code(404);
                return ['error' => sprintf('Unable to load cart by card id "%d".', $order_id)];
            }
    
            $resultData = array();
    
            foreach ($checkOutInformation['requested_data'] as $data => $allowed)
            {
                if ($allowed){
                    switch ($data)
                    {
                        case $data == 'client_data':
                            $resultData = array_merge($resultData, $this->getClientData($user_id));
                            break;
                        case $data == 'historical':
                            try{
                                $resultData = array_merge($resultData, $this->getHistoricalData($user_id));
                            } catch (Exception $e) {}
                            break;
                        case $data == 'lost_carts':
                            try{
                                $resultData = array_merge($resultData, $this->getLostCartsData($user_id, $order_id));
                            } catch (Exception $e) {}
                            break;
                        case $data == 'images':
                            $resultData = array_merge($resultData, $this->getImagesData($order_id));
                            break;
                        default :
                            break;
                    }
                }
            }
    
            return $resultData;
        }

        private function getClientData($user_id)
        {
            if (!$user_id) {
                return array('client_data' => array());
            }

            $customer = new WC_Customer( $user_id );

            if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                $clientData = array(
                    'entity_id' => intval($customer->id),
                    'email' => $customer->email,
                    'created_at' => $customer->date_created,
                    'updated_at' => $customer->date_modified,
                    'is_active' => true,
                    'first_name' => $customer->first_name,
                    'last_name' => $customer->last_name
                );
            } else {
                $clientData = array(
                    'entity_id' => intval($customer->get_id()),
                    'email' => $customer->get_email(),
                    'created_at' => $customer->get_date_created()->format('Y-m-d H:i:s'),
                    'updated_at' => $customer->get_date_modified()->format('Y-m-d H:i:s'),
                    'is_active' => true,
                    'first_name' => $customer->get_first_name(),
                    'last_name' => $customer->get_last_name()
                );
            }
            
            return array('client_data' => $clientData);
        }

        private function getHistoricalData($user_id)
        {
            if (!$user_id) {
                return array('historical' => array());
            }
            
            $query = array(
                'customer_id' => $user_id,
                'status' => 'completed'
            );

            $orders = wc_get_orders( $query );
    
            $historicalData = array();
            foreach ($orders as $order) {
                $items;
                $orderData = array();

                if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                    $orderData = array(
                        'entity_id' => intval($order->id),
                        'increment_id' => intval($order->id),
                        'state' => $order->status,
                        'customer_id' => intval($order->user->id),
                        'subtotal' => (float)round($order->subtotal, 2),
                        'grand_total' => (float)round($order->total, 2),
                        'discount_amount' => (float)round($order->discount_total,2 ),
                        'shipping_amount' => (float)round($order->shipping_total, 2),
                        'created_at' => $order->date_created,
                        'updated_at' => $order->date_modified,
                        'products' => array()
                    );
                    $items = $order->get_items();
                } else {
                    $orderData = array(
                        'entity_id' => intval($order->get_id()),
                        'increment_id' => intval($order->get_id()),
                        'state' => $order->get_status(),
                        'customer_id' => intval($order->get_user_id()),
                        'subtotal' => (float)round($order->get_subtotal(), 2),
                        'grand_total' => (float)round($order->get_total(), 2),
                        'discount_amount' => (float)round($order->get_discount_total(),2 ),
                        'shipping_amount' => (float)round($order->get_shipping_total(), 2),
                        'created_at' => $order->get_date_created()->format('Y-m-d H:i:s'),
                        'updated_at' => $order->get_date_modified()->format('Y-m-d H:i:s'),
                        'products' => array()
                    );
                    $items = $order->get_items();
                }
    
                foreach ($items as $item) {
                    if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                        $itemData = array(
                            // 'item_id' => intval($item->id),
                            // 'order_id' => intval($item->order->id),
                            'product_id' => intval($item->product->id),
                            'name' => $item->name,
                            'price' => (float)round($item->total, 2),
                            'quantity' => (float)round($item->quantity, 2)
                        );
                    } else {
                        $itemData = array(
                            // 'item_id' => intval($item->get_id()),
                            // 'order_id' => intval($item->get_order_id()),
                            'product_id' => intval($item->get_product_id()),
                            'name' => $item->get_name(),
                            'price' => (float)round($item->get_total(), 2),
                            'quantity' => (float)round($item->get_quantity(), 2)
                        );
                    }

                    array_push($orderData['products'], $itemData);
                }

                array_push($historicalData, $orderData);
            }
    
            return array('historical' => $historicalData);
        }

        private function getLostCartsData($user_id, $order_id)
        {
            if (!$user_id) {
                return array('lost_carts' => array());
            }

            $query = array(
                'customer_id' => $user_id,
                'status' => array('pending', 'on-hold', 'failed')
            );

            $orders = wc_get_orders( $query );
    
            $lostCartData = array();
    
            foreach ($orders as $order) {
                if ($order->id == $order_id) {
                    continue;
                }

                $cartData = array();
                if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                    $cartData = array(
                        'entity_id' => intval($order->id),
                        'created_at' => $order->date_created,
                        'updated_at' => $order->date_modified,
                        'products' => array()
                    );
                    $items = $order->get_items();
                } else {
                    $cartData = array(
                        'entity_id' => intval($order->get_id()),
                        'created_at' => $order->get_date_created()->format('Y-m-d H:i:s'),
                        'updated_at' => $order->get_date_modified()->format('Y-m-d H:i:s'),
                        'products' => array()
                    );
                    $items = $order->get_items();
                }
    
                if (!$items) continue;
    
                foreach ($items as $item){
                    if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                        $itemData = array(
                            'product_id' => intval($item->product->id),
                            'price' => (float)round($item->total, 2),
                            'quantity' => (float)round($item->quantity, 2)
                        );
                    } else {
                        $itemData = array(
                            'product_id' => intval($item->get_product_id()),
                            'price' => (float)round($item->get_total(), 2),
                            'quantity' => (float)round($item->get_quantity(), 2)
                        );
                    }

                    array_push($cartData['products'], $itemData);
                }
    
                array_push($lostCartData, $cartData);
            }
    
            return array('lost_carts' => $lostCartData);
        }

        private function getImagesData($order_id) {
            $imagesArray = array();

            $order = wc_get_order( $order_id );
            $items = $order->get_items();
    
            foreach ($items as $item) {
                $product_id = '';
                if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                    $product_id = intval($item->product->id);
                } else {
                    $product_id = intval($item->get_product_id());
                }

                $product = wc_get_product( $product_id );
                $image_id = '';
                if ( version_compare( WOOCOMMERCE_VERSION, '3.0', '<' ) ) {
                    $image_id = $product->image->id;
                } else {
                    $image_id = $product->get_image_id();
                }
                $image = wp_get_attachment_image_url($image_id, 'full');

                $image64 = base64_encode(file_get_contents($image));

                array_push($imagesArray, array(
                    'id' => $product_id,
                    'imageB64' => $image64
                ));
            }

            if ($imagesArray != ''){
                return array('images' => $imagesArray);
            } else {
                return array('images' => array());
            }
        }

    } // end \WC_Gateway_Pepper class

    class WC_Gateway_Pepper_N extends WC_Gateway_Pepper {
        /**
         * Constructor for the gateway.
         */
        public function __construct() {
            parent::__construct(FALSE);
            $this->id = 'pepper_n_gateway';

            $this->title = $this->title_n;
            $this->description = $this->description_n;
            $this->is_standard = false;

            //Actions
            add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
            add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );
                
            if ( version_compare( WOOCOMMERCE_VERSION, '2.0', '<' ) ) {
                // Check for gateway messages using WC 1.X format
                add_action( 'init', array( $this, 'check_notification' ) );
                add_action( 'woocommerce_update_options_payment_gateways', array( &$this, 'process_admin_options' ) );
            } else {
                // Payment listener/API hook (WC 2.X)
                add_action( 'woocommerce_api_' . strtolower( get_class( $this ) ), array( $this, 'check_notification' ) );
                add_action('woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
            }

            add_action( 'woocommerce_receipt_'.$this->id, array( $this, 'receipt_page' ) );

            //Filter: if pepper_n is disabled then remove from payment_gateways array.
            add_filter( 'woocommerce_available_payment_gateways', array( $this, 'pepper_n_disable_manager' ) );
        }

        function get_pepper_client() {
            $settings = get_option('woocommerce_pepper_gateway_settings', null);
            $public_key = $settings[ 'public_key' ];
            $secret_key = $settings[ 'secret_key' ];
            $api_url = $settings[ 'api_url' ];
            $api_key = $settings[ 'api_key' ];
            return new Pepper\Client($public_key, $secret_key, $api_url);
        }

        function pepper_n_disable_manager($available_gateways) {
            if ( $this->enabled_n == 'no' ){
                unset( $available_gateways[$this->id] );
            }
            return $available_gateways;
        }
    }//end \WC_Gateway_Pepper_N class
}

$GLOBALS['pepper'] = new WC_Pepper();
